#include <stdlib.h>
#include <string.h>
#include <fnmatch.h>
#include <glob.h>

#include "model/attribute_list.h"

//-------------------------------- private function definition -------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------- private attributes ----------------------------------------------------------------
static kms_attribute_t *first_in_list=NULL;

//--------------------------------------------------------------------------------------------------------------------

//-------------------------------- public functions ------------------------------------------------------------------
error_code_t attribute_list_new_attribute_no_credentials(
		kms_attribute_t **attr_ptr, const char *sysfs_path, const char *name) {
	return attribute_list_new_attribute (attr_ptr, sysfs_path, name, NULL);
}


error_code_t attribute_list_new_attribute(kms_attribute_t **attr_ptr, const char *sysfs_path,
		const char *name, const kms_credentials_t *credentials)
{
	kms_attribute_t *attribute;
	size_t size;

	size=sizeof(kms_attribute_t)+strlen(sysfs_path)+1+strlen(name)+1;
	attribute=malloc(size);
	if (attribute==NULL)
		return RESULT_NORESOURCES;

	strcpy(attribute->path, sysfs_path);

	/* point to \0 of sysfs path */
	attribute->sepchar_ptr = attribute->path + strlen(attribute->path);
	/* replace by / */
	*(attribute->sepchar_ptr) = (char)SEPCHAR_ATTR_PATH;
	*(attribute->sepchar_ptr+1) = '\0';

	/* append separation character and attribute */
	strcat(attribute->path, name);

	attribute->apply_credentials=(credentials!=NULL);
	if (credentials!=NULL)
		memcpy(&attribute->credentials,credentials,sizeof(kms_credentials_t));
	attribute->next_in_list_ptr=NULL;
	attribute->previous_in_list_ptr=NULL;

	*attr_ptr=attribute;
	return RESULT_OK;
}

void attribute_list_free_attribute(kms_attribute_t *attribute)
{
	free(attribute);
}

error_code_t attribute_list_add(kms_attribute_t *attribute)
{
	error_code_t result = RESULT_OK;

	//--- add it to the list ---
	//update forward pointer
	attribute->next_in_list_ptr=first_in_list;
	first_in_list=attribute;

	//update backward pointer
	if (attribute->next_in_list_ptr!=NULL)
		attribute->next_in_list_ptr->previous_in_list_ptr=attribute;

	return result;
}

void attribute_list_remove_and_free(kms_attribute_t *attribute)
{
	kms_attribute_t *previous;
	kms_attribute_t *next;

	previous=attribute->previous_in_list_ptr;
	next=attribute->next_in_list_ptr;

	if (previous!=NULL)
		//we have a previous element -> link it forward to our next
		previous->next_in_list_ptr=next;
	else
		//we are the first in the list -> make our next one the first in the list
		first_in_list=next;

	//update the backwards link of the elemnt succeeding us of there is any
	if (next!=NULL)
		next->previous_in_list_ptr=previous;

	//remove us from memory
	attribute_list_free_attribute(attribute);
}

bool attribute_list_is_empty(void)
{
	return first_in_list==NULL;
}

void attribute_list_clear(void)
{
	kms_attribute_t *tmp;

	while (first_in_list!=NULL)
	{
		tmp=first_in_list;
		first_in_list=tmp->next_in_list_ptr;
		attribute_list_free_attribute(tmp);
	}
}

kms_attribute_t *attribute_list_get_first_attribute(void)
{
	return first_in_list;
}

kms_attribute_t *attribute_list_get_next_attribute(kms_attribute_t *attribute)
{
	if (attribute==NULL)
		return NULL;
	return attribute->next_in_list_ptr;
}

kms_attribute_t *attribute_list_get_first_attribute_filtered(const char *sysfs_path_base)
{
	kms_attribute_t *attribute;
	char* sysfs_path;

	attribute=first_in_list;
	while (attribute!=NULL)
	{
		sysfs_path = attribute_get_sysfs_path(attribute);
		if (fnmatch(sysfs_path, sysfs_path_base, FNM_PATHNAME|FNM_NOESCAPE)==0)
			return attribute;
		attribute=attribute->next_in_list_ptr;
	}

	return NULL;
}

kms_attribute_t *attribute_list_get_next_attribute_filtered(kms_attribute_t *attribute, const char *sysfs_path_base)
{
	char* sysfs_path;

	//move one forward if possible
	if (attribute!=NULL)
		attribute=attribute->next_in_list_ptr;

	while (attribute!=NULL)
	{
		sysfs_path = attribute_get_sysfs_path(attribute);
		if (fnmatch(sysfs_path, sysfs_path_base, FNM_PATHNAME|FNM_NOESCAPE)==0)
			return attribute;
		attribute=attribute->next_in_list_ptr;
	}

	return NULL;
}

bool attribute_list_is_attribute_available(kms_attribute_t *attribute)
{
	bool result = false;
	int gl_result;
	glob_t gl;
	char *attr_path = attribute_get_attr_path(attribute);

	gl_result=glob (attr_path, GLOB_NOSORT | GLOB_ERR, NULL, &gl);
	globfree(&gl);

	if (gl_result==0)
		result = true;

	return result;
}

char* attribute_get_sysfs_path(kms_attribute_t *attribute) {
	*(attribute->sepchar_ptr) = (char)SEPCHAR_SYSFS_PATH;
	return attribute->path;
}

char* attribute_get_attr_path(kms_attribute_t *attribute) {
	*(attribute->sepchar_ptr) = (char)SEPCHAR_ATTR_PATH;
	return attribute->path;
}

//--------------------------------------------------------------------------------------------------------------------

//-------------------------------- private functions -----------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------------------
